// @flow
import React, {Fragment} from 'react';
import type {Node} from 'react';
import classNames from 'classnames';
import Icon from '../Icon';
import SingleSelect from '../SingleSelect';
import blockStyles from './block.scss';

type Props = {
    activeType?: string,
    children: Node,
    dragHandle?: Node,
    expanded: boolean,
    icons?: Array<string>,
    onCollapse: () => void,
    onExpand: () => void,
    onRemove?: () => void,
    onSettingsClick?: () => void,
    onTypeChange?: (type: string | number) => void,
    types?: {[key: string]: string},
};

export default class Block extends React.Component<Props> {
    static defaultProps: {
        expanded: false,
    };

    handleCollapse = () => {
        const {expanded, onCollapse} = this.props;
        if (expanded) {
            onCollapse();
        }
    };

    handleExpand = () => {
        const {expanded, onExpand} = this.props;
        if (!expanded) {
            onExpand();
        }
    };

    handleTypeChange = (type: string | number) => {
        const {onTypeChange} = this.props;

        if (onTypeChange) {
            onTypeChange(type);
        }
    };

    render() {
        const {activeType, children, dragHandle, expanded, icons, onSettingsClick, onRemove, types} = this.props;

        const blockClass = classNames(
            blockStyles.block,
            {
                [blockStyles.expanded]: expanded,
            }
        );

        return (
            <section className={blockClass} onClick={this.handleExpand} role="switch">
                <div className={blockStyles.handle}>
                    {dragHandle}
                </div>
                <div className={blockStyles.content}>
                    <header className={blockStyles.header}>
                        {expanded
                            ? <Fragment>
                                {types && Object.keys(types).length > 1 &&
                                    <div className={blockStyles.types}>
                                        <SingleSelect onChange={this.handleTypeChange} value={activeType}>
                                            {Object.keys(types).map((key) => (
                                                <SingleSelect.Option key={key} value={key}>
                                                    {types[key]}
                                                </SingleSelect.Option>
                                            ))}
                                        </SingleSelect>
                                    </div>
                                }
                                {icons &&
                                    <div className={blockStyles.icons}>
                                        {icons.map((icon) => <Icon key={icon} name={icon} />)}
                                    </div>
                                }
                                <div className={blockStyles.iconButtons}>
                                    {onSettingsClick && <Icon name="su-cog" onClick={onSettingsClick} />}
                                    {onRemove && <Icon name="su-trash-alt" onClick={onRemove} />}
                                    <Icon name="su-angle-up" onClick={this.handleCollapse} />
                                </div>
                            </Fragment>
                            : <Fragment>
                                {icons &&
                                    <div className={blockStyles.icons}>
                                        {icons.map((icon) => <Icon key={icon} name={icon} />)}
                                    </div>
                                }
                                {types && activeType && <div className={blockStyles.type}>{types[activeType]}</div>}
                                <Icon name="su-angle-down" />
                            </Fragment>
                        }
                    </header>
                    <article className={blockStyles.children}>{children}</article>
                </div>
            </section>
        );
    }
}
