<?php

namespace PHPCR\Observation;

/**
 * A storage object for event filter configuration. A blank EventFilter is
 * acquired through {@link ObservationManager::createEventFilter()}.
 *
 * The parameters of the filter can then be set by chaining the set methods,
 * since each method returns the same EventFilter with the indicated parameter
 * set.
 *
 * Once the filter is configured, it and an {@link EventListenerInterface}
 * object are passed to {@link ObservationManager::addEventListener()}.
 *
 * The filter restricts which events are sent to the EventListener according to
 * the following parameters. Note that the term <i>associated parent node</i>
 * of an event means the parent node of the item at (or formerly at) the path
 * returned by {@link EventInterface::getPath()}.
 * <li>
 * eventTypes: A bitwise OR of the event types to be listened to. See
 * {@link EventInterface} for details.
 * </li>
 * <li>
 * absPath, isDeep: Only events whose associated parent node is at absPath (or
 * within its subgraph, if isDeep is true) will be received. It is permissible
 * to register a listener for a path where no node currently exists.
 * </li>
 * <li>
 * uuid: Only events whose associated parent node has one of the identifiers in
 * this list will be received. If his parameter is null then no
 * identifier-related restriction is placed on events received. Note that
 * specifying an empty array instead of null would result in no nodes being
 * listened to. The term "UUID" is used for compatibility with JCR 1.0.
 * </li>
 * <li>
 * nodeTypes: Only events whose associated parent node has one of the node
 * types (or a subtype of one of the node types) in this list will be received.
 * If his parameter is null then no node type-related restriction is placed on
 * events received. Note that specifying an empty array instead of null would
 * result in no nodes types being listened to.
 * </li>
 * <li>
 * noLocal: if true, then events generated by the session through which the
 * listener was registered are ignored. Otherwise, they are not ignored.
 * </li>
 * </ul>
 * The restrictions are "ANDed" together. In other words, for a particular node
 * to be "listened to" it must meet all the restrictions.
 *
 * @license http://www.apache.org/licenses Apache License Version 2.0, January 2004
 * @license http://opensource.org/licenses/MIT MIT License
 *
 * @api
 *
 * @since JCR 2.1
 *
 */
interface EventFilterInterface
{
    /**
     * Sets the eventTypes parameter of the filter.
     *
     * If left unset, this parameter defaults to null.
     *
     * @param int $eventTypes an int.
     *
     * @return EventFilterInterface this filter object with the eventTypes parameter set.
     */
    public function setEventTypes($eventTypes);

    /**
     * Returns the eventTypes parameter of the filter.
     *
     * @return int.
     */
    public function getEventTypes();

    /**
     * Sets the absPath parameter of the filter.
     *
     * If left unset, this parameter defaults to null.
     *
     * @param string $absPath an absolute path String.
     *
     * @return EventFilterInterface this filter object with the absPath parameter set.
     */
    public function setAbsPath($absPath);

    /**
     * Returns the absPath parameter of the filter.
     *
     * @return string
     */
    public function getAbsPath();

    /**
     * Sets the isDeep parameter of the filter.
     *
     * If left unset, this parameter defaults to false.
     *
     * @param boolean $isDeep
     *
     * @return EventFilterInterface this filter object with the absPath parameter set.
     */
    public function setIsDeep($isDeep);

    /**
     * Returns the isDeep parameter of the filter.
     *
     * @return boolean
     */
    public function getIsDeep();

    /**
     * Sets the identifiers parameter of the filter.
     *
     * If left unset, this parameter defaults to null.
     *
     * @param string[] $identifiers
     *
     * @return EventFilterInterface this filter object with the absPath parameter set.
     */
    public function setIdentifiers(array $identifiers);

    /**
     * Returns the uuids parameter of the filter.
     *
     * @return string[]
     */
    public function getIdentifiers();

    /**
     * Sets the nodeTypes parameter of the filter.
     *
     * If left unset, this parameter defaults to null.
     *
     * @param string[] $nodeTypes names of the node types to filter on
     *
     * @return EventFilterInterface this filter object with the absPath parameter set.
     */
    public function setNodeTypes(array $nodeTypes);

    /**
     * Returns the nodeTypes parameter of the filter.
     *
     * @return string[]
     */
    public function getNodeTypes();

    /**
     * Sets the noLocal parameter of the filter.
     *
     * If left unset, this parameter defaults to false.
     *
     * @param boolean $noLocal
     *
     * @return EventFilterInterface this filter object with the absPath parameter set.
     */
    public function setNoLocal($noLocal);

    /**
     * Returns the noLocal parameter of the filter.
     *
     * @return boolean.
     */
    public function getNoLocal();
}
